// Proramm für das ArduSmartPilot Interface zum Flug-Modell-Simulator (FMS)
// Dieses Programm läuft auf einem getrennten Arduino, der via USB seriell mit dem PC kommuniziert.
// Der Arduino empfängt die drei Servosteuersignale (PWM-Signale) des Arduino am Ardusmartpilot und
// misst diese aus und generiert daraus Steuerdaten für den FMS. Am FMS muss unter dem Menüpunkt
// Steuerung "Analoge Steuerung" > "Serielles PIC-Interface" ausgewählt werden. Weiter muss der COM
// Port im FMS ausgewählt werden, mit dem der Arduino verbunden ist. Da im FMS die Auswahl nur bis
// COM Port 6 möglich ist, der Arduino oft aber automatisch von Windows einen viel höheren COM Port
// zugewiesen bekommt, muss am PC in der Systemsteuerung im Gerätemanager > Anschlüsse (COM&LPT) der 
// Anschluss des Arduino ausgewält und dieser auf einen COM Port <7 geändert werden.
// Funktioniert die serielle Kommunikation, dann muss im FMS unter Steuerung "Belegung/Kalibration" 
// für das Seiten-, Höhenruder und die Motorleistung (=Motordrosse) die Kanäle 1,2 und 3 eingegeben werden.
// Abschließend werden diese drei Kanäle dann kalibiert (siehe Anweisungen im FMS).
// Wenn der tatsächliche Ruderauschlag im Simulator verkehrt herum ist, dann kann dieser im Menü
// Steuerung invertiert werden (ist z.B. beim Seitenruder notwendig)

// Originaler Code von Dudu Zanette.
// Siehe: http://duduzanette.com/others/simulator_cable_simple.ino

#define RX_SIGNAL_TIMEOUT 15000 // 15mS

//Define Channel/Input association
// Kanal 1 wird auf Digitalpin 2 gelegt, Kanal 2 auf Digitalpin 3 usw.
#define CHANNEL1_INPUT 2
// Beim ArduSmartPilot werden nur drei Kanäle verwendet: 
#define NUM_OF_RXCHANNELS 3

// data for each channel
byte channelData[NUM_OF_RXCHANNELS];

// order that the channels are sent
// Die Zuordnung der Kanäle zu den Steuerfunktionen ist im FMS unter 
// "Steuerung - Analoge Steuerung - Serielles PIC-Interface" einsehbar
// bzw. auch einstellbar: Die richtigen Einstellungen sind
// Seitenruder 1, Höhenruder 2, Motordrossel (=Motorleistung) 3
// Vom Arduino des ArduSmartpilot werden die Servosignale wie folgt auf diesen
// Interface-Arduino übertragen: Seitenruder (Pin D9) auf Pin D2, Höhenruder (Pin D10) 
// auf Pin D3, Motorleistung (Pin D11) auf Pin D4.

// Hier wird festgelegt, in welcher Reihenfolge die Digitalpins abgefragt werden. Damit
// das Programm nicht endlos in einer While-Schleife bleibt, müssen alle diese Pins mit
// einem PWM-Signal belegt sein.
unsigned int channelOrder[8] = {0, 1, 2, 3, 4, 5, 6, 7};

void setup()
{
  Serial.begin(9600);

  for(int i = 0; i < NUM_OF_RXCHANNELS; i++)
  {
    // Die digitalen Eingänge konfigurieren
    pinMode(i+CHANNEL1_INPUT,INPUT);
  }
}

// divide t by 10 and keep the
// output between 70 and 250
// Wieso ist das nötig?
byte convertData(unsigned long t)
{
  if(t > 2500L)
  {
    return 250;
  }
  else if(t < 700L)
  {
    return 70;
  }
  
  return (byte)(t / 10);
}

void loop()
{
  for(int c = 0; c < NUM_OF_RXCHANNELS; c++)
  {
    unsigned int channel = channelOrder[c];
    
    // read pulse duration
    while(digitalRead(channel+CHANNEL1_INPUT)==LOW);
    unsigned long t = micros();
    while(digitalRead(channel+CHANNEL1_INPUT)==HIGH);
    unsigned long duration = micros()-t;
    
    // Gleitender Mittelwert: Der vorherige Wert wird mit 3 multipliziert,
    // der aktuelle Wert dazu addiert, und dann durch 4 geteilt. Das
    // Ergebnis wird dann in den Wertearray channelData eingetragen und
    // dient beim nächsten Durchlauf als neuer vorheriger Wert.
    // using average of four readings
    // slower response but with less noise
    unsigned int data = channelData[channel] << 1;
    data += channelData[channel];
    data += convertData(duration);
    channelData[channel] = (byte)(data>>2);
  }

  // send PPM data to FMS
  // 0xF0 + NUM_OF_RXCHANNELS
  Serial.write(0xF0+NUM_OF_RXCHANNELS);  
  
  // 0x00
  Serial.write((byte)0x00);
  
  // channel info
  for(int channel = 0; channel < NUM_OF_RXCHANNELS; channel++)
  {
    Serial.write(channelData[channel]);
  }
}
